const fs = require('fs');
const path = require('path');

const parseLogFile = (logFilePath) => {
  try {
    const logContent = fs.readFileSync(logFilePath, 'utf8');
    const lines = logContent.split('\n').filter(line => line.trim() !== '');
    
    return lines.map((line, index) => {
      // Parse different log formats
      const parsed = parseLogLine(line);
      return {
        id: index + 1,
        raw: line,
        ...parsed
      };
    }).reverse(); // Most recent first
  } catch (error) {
    console.error('Error reading log file:', error);
    return [];
  }
};

const parseLogLine = (line) => {
  // Try to parse JSON format first (Winston JSON format)
  try {
    const jsonLog = JSON.parse(line);
    return {
      timestamp: jsonLog.timestamp || new Date().toISOString(),
      level: jsonLog.level || 'info',
      message: jsonLog.message || line,
      meta: jsonLog.meta || {},
      format: 'json'
    };
  } catch (e) {
    // If not JSON, try to parse common log formats
    return parseTextLogLine(line);
  }
};

const parseTextLogLine = (line) => {
  // Try to extract timestamp, level, and message from text format
  // Format: YYYY-MM-DD HH:mm:ss [LEVEL] message
  const timestampRegex = /^(\d{4}-\d{2}-\d{2}[T\s]\d{2}:\d{2}:\d{2}(?:\.\d{3})?(?:Z|[+-]\d{2}:\d{2})?)/;
  const levelRegex = /\[(error|warn|warning|info|debug|verbose|silly)\]/i;
  
  let timestamp = null;
  let level = 'info';
  let message = line;
  
  // Extract timestamp
  const timestampMatch = line.match(timestampRegex);
  if (timestampMatch) {
    timestamp = timestampMatch[1];
    message = line.substring(timestampMatch[0].length).trim();
  }
  
  // Extract level
  const levelMatch = message.match(levelRegex);
  if (levelMatch) {
    level = levelMatch[1].toLowerCase();
    message = message.replace(levelMatch[0], '').trim();
  }
  
  // If no timestamp found, use current time
  if (!timestamp) {
    timestamp = new Date().toISOString();
  }
  
  return {
    timestamp,
    level,
    message,
    format: 'text'
  };
};

const getLevelColor = (level) => {
  const colors = {
    error: '#dc3545',
    warn: '#ffc107',
    warning: '#ffc107',
    info: '#17a2b8',
    debug: '#6c757d',
    verbose: '#6f42c1',
    silly: '#e83e8c'
  };
  return colors[level] || '#6c757d';
};

const getLevelIcon = (level) => {
  const icons = {
    error: '❌',
    warn: '⚠️',
    warning: '⚠️',
    info: 'ℹ️',
    debug: '🐛',
    verbose: '📝',
    silly: '🎭'
  };
  return icons[level] || 'ℹ️';
};

const formatTimestamp = (timestamp) => {
  try {
    const date = new Date(timestamp);
    return date.toLocaleString();
  } catch (e) {
    return timestamp;
  }
};

module.exports = {
  parseLogFile,
  getLevelColor,
  getLevelIcon,
  formatTimestamp
};
