const { Router } = require("express");
const { initiateVoting } = require("../controllers/voteController.js");
const { Nomination } = require("../models/nomination.js");
const { Category } = require("../models/category.js");
const { vSession } = require("../models/voteSession.js");
const { event_profile } = require("../models/config.js");
const logger = require("../utilities/logger.js");
const { Vote } = require("../models/vote.js");

const voteRoutes = Router();
const pubkey = process.env.paystack_public_key;
voteRoutes.post("/create", initiateVoting);

voteRoutes.get("/:id", async (req, res) => {
  const id = req.params.id;
  const status = await event_profile.findOne();
  const event = status;

  if (status.voting === true) {
    try {
      const candidate = await Nomination.findById(id);
      if (!candidate) {
        return res.status(404).send("Candidate not found.");
      }
      const rate = status.rate;
      //console.log(rate)
      return res.render("vote", { candidate, rate, pubkey, event });
    } catch (error) {
      console.error("Error fetching candidate:", error);
      if (error.kind === "ObjectId") {
        return res.status(400).send("Invalid candidate ID.");
      }
      return res.status(500).send("Server error. Please try again later.");
    }
  } else {
    return res.send(`
      <html>
      <head>
        <title>Voting Closed</title>
        <style>
          body {
            font-family: Arial, sans-serif;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            margin: 0;
            background-color: #f0f0f0;
          }
          .message-container {
            text-align: center;
            padding: 20px;
            background: #fff;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
          }
          h1 {
            color: #333;
          }
          p {
            color: #666;
          }
        </style>
      </head>
      <body>
        <div class="message-container">
          <h1>Voting Closed</h1>
          <p>Unfortunately, voting is currently closed. Please check back later.</p>
        </div>
      </body>
      </html>
    `);
  }
});

voteRoutes.post("/session", async (req, res) => {
  const { contact, votes, code, category, reference } = req.body;
  
  if (!contact?.trim() || !votes || !code?.trim() || !category?.trim() || !reference?.trim()) {
    return res.status(400).json({
        status: "Failed",
        message: 'All parameters needed'
    });
}

  // Create session object
  const session = {
    contact,
    votes,
    nominee: code,
    category,
    reference
  };

  try {
    // Save the session to the database
    const newSession = new vSession(session);
    await newSession.save();
    
    return res.status(201).json({
      status: "Success",
      message: "Session saved successfully!"
    });
  } catch (error) {
    // Handle any errors that occur during session saving
    console.error(error);
    return res.status(500).json({
      status: "Failed",
      message: "An error occurred while saving the session.",
      error: error.message
    });
  }
});

voteRoutes.post('/case', (req, res) => {
  logger.info(JSON.stringify(req.body)); // Pretty print the JSON for better readability
  return res.json({
    status: 'success',
    message: 'Case lodged successfully',
    data: req.body
  });
});

voteRoutes.get('/results/:categoryId', async (req, res) => {
  try {
     const event = await event_profile.findOne();
    if(!event.public_viewing){
      return res.send(`
         <html>
      <head>
        <title>Voting Closed</title>
        <style>
          body {
            font-family: Arial, sans-serif;
            display: flex;
            justify-content: center;
            align-items: center;
            height: 100vh;
            margin: 0;
            background-color: #f0f0f0;
          }
          .message-container {
            text-align: center;
            padding: 20px;
            background: #fff;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
          }
          h1 {
            color: #333;
          }
          p {
            color: #666;
          }
        </style>
      </head>
      <body>
        <div class="message-container">
          <h1>Results Closed</h1>
          <p>Unfortunately, Public viewing of Results is currently closed. Please check back later.</p>
        </div>
      </body>
      </html>
        `)
    }
      const category = req.params.categoryId;
      const Cat = await Category.findById(category);
      const votes = await Vote.countDocuments({ category: category });
  
      const pipeline = [
        {
          $match: {
            category: Cat._id,
            isApproved: true,
          },
        },
        {
          $lookup: {
            from: "votes",
            localField: "_id",
            foreignField: "nominee",
            as: "votes",
          },
        },
        {
          $addFields: {
            category: { $ifNull: ["$category", "No Category"] },
          },
        },
        {
          $lookup: {
            from: "categories",
            localField: "category",
            foreignField: "_id",
            as: "category",
          },
        },
        {
          $unwind: {
            path: "$category",
            preserveNullAndEmptyArrays: true,
          },
        },
        {
          $addFields: {
            category: "$category.name",
          },
        },
        {
          $addFields: {
            votes: { $size: "$votes" },
          },
        },
        {
          $project: {
            _id: 1,
            name: 1,
            imageUrl: 1,
            category: 1,
            votes: 1,
            code: 1,
          },
        },
        {
          $sort: {
            votes: -1, // Sort by votes in descending order
          },
        },
      ];
  
      const Candidates = await Nomination.aggregate(pipeline);
  
      res.render("dashboard", { Candidates, Cat, votes });
  } catch (error) {
    console.error("Error fetching results:", error);
    res.status(500).json({ message: "Error fetching results" });
  }
});

module.exports = { voteRoutes };
