const { Router } = require("express");
const { verifyPassword, hashPassword } = require("../utilities/encryption.js");
const { User } = require("../models/users.js");
const { validKey, getConfig } = require("../utilities/appConfig.js");
const multer = require("multer");
const { requireLogin } = require("../middleware/authMiddleware.js");
const { Nomination } = require("../models/nomination.js");
const { Category } = require("../models/category.js");
const { Vote } = require("../models/vote.js");
const { event_profile } = require("../models/config.js");
const FormData = require("form-data");
const axios = require("axios");
const upload = multer();

const isValidKey = (key) => key === validKey;

const isValidTargets = (targets) =>
  Array.isArray(targets) && targets.length > 0;

// Function to check if a user already exists
async function userExists(username) {
  try {
    const user = await User.findOne({ username: username });
    return !!user; // Convert user to boolean (true if found, false if not)
  } catch (error) {
    console.error("Error checking user existence:", error);
    return; // Assume user doesn't exist on error
  }
}

const userRoutes = Router();

userRoutes.get("/login", (req, res) => {
  if (req.session.authenticated) {
    // Authentication successful, you can redirect or send a success message
    return res.redirect("/u/profile");
  }
  return res.render("login");
});

userRoutes.get("/logout", (req, res) => {
  req.session.destroy((err) => {
    if (err) {
      console.error("Error destroying session:", err);
    }
  });
  res.redirect("/u/login");
});

userRoutes.post("/login", upload.none(), async (req, res) => {
  try {
    // Get the submitted form data
    const { email, password } = req.body;

    // Find the user with the provided email
    const current_user = await User.findOne({ email: email });

    if (!current_user) {
      // User not found
      return res.status(401).json("User not found");
    }

    // Verify the password
    const hashedPassword = current_user.password;
    const isMatch = await verifyPassword(password, hashedPassword);

    // Check if the password matches
    if (!isMatch) {
      // Incorrect password
      return res.status(401).json("Incorrect password");
    }

    // Mark the session as authenticated
    req.session.authenticated = true;
    req.session.user = current_user;

    const profile = await event_profile.findOne();
    if (!profile) {
      console.log("Profile not found, using default profile");
      return res.redirect("/config");
    }

    res.redirect("/u/profile");
  } catch (error) {
    console.error(error);
    res.status(500).send("Internal Server Error");
  }
});

userRoutes.post("/signup/:key", upload.single("file"), async (req, res) => {
  const key = req.params.key;
  if (!isValidKey(key)) {
    return res.status(401).send("Invalid request key");
  }

  const Users = req.body;

  if (!Array.isArray(Users)) {
    return res
      .status(400)
      .send("Invalid request body, expected an array of users");
  }

  const results = [];

  for (const current_user of Users) {
    const { name, username, password, email, profile_pic } = current_user;

    if (!name || !username || !password || !email) {
      results.push({
        username,
        status: "error",
        message: "Missing required fields",
      });
      continue;
    }

    if (!profile_pic) {
      current_user.profile_pic = "https://iili.io/dKSXTua.md.png";
    }

    try {
      const userExistsCheck = await userExists(username);
      if (userExistsCheck) {
        results.push({
          username,
          status: "error",
          message: "User already exists",
        });
        continue;
      }

      // Create a new user instance
      let new_user = new User(current_user);

      // Save the new user to the database
      await new_user.save();

      results.push({
        username,
        status: "success",
        message: `User ${new_user.name} added successfully`,
      });
    } catch (error) {
      console.error(error);
      results.push({
        username,
        status: "error",
        message: "Internal Server Error",
      });
    }
  }

  res.json(results);
});

userRoutes.get("/profile", requireLogin, async (req, res) => {
  try {
    let current_user = req.session.user;
    current_user = await User.findOne({ email: current_user.email });
    if (!current_user) {
      return res.status(404).send("User not found");
    }
    let { name, email, profile_pic } = current_user;
    let nominations = (await Nomination.find()).length;
    let categories = await Category.find();
    let votes = await Vote.countDocuments();
    const event = await getConfig();
    let earnings = votes * event.rate * event.income;
    earnings = earnings.toFixed(2); // Rounds to two decimal places and converts to a string

    res.render("profile", {
      name,
      email,
      profile_pic,
      nominations,
      categories,
      event,
      earnings,
    });
  } catch (error) {
    console.error(error);
    res.status(500).send("Internal Server Error");
  }
});

userRoutes.get("/update", requireLogin, (req, res) => {
  res.render("update");
});

userRoutes.post(
  "/update",
  requireLogin,
  upload.single("profile_pic"),
  async (req, res) => {
    const { name, username, email, old_pass, new_pass, c_pass } = req.body;
    const profilePic = req.file;

    try {
      const current_user = req.session.user;
      const updates = {};

      // Handle profile picture upload if a file is provided
      if (profilePic) {
        const form = new FormData();
        form.append("key", "6d207e02198a847aa98d0a2a901485a5");
        form.append("action", "upload");
        form.append("source", profilePic.buffer, profilePic.originalname);
        form.append("format", "json");

        const response = await axios.post(
          "https://freeimage.host/api/1/upload",
          form,
          {
            headers: form.getHeaders(),
          }
        );

        const imageUrl = response.data.image.url;
        updates.profile_pic = imageUrl;
      }

      // Handle password change if old_pass and new_pass are provided
      if (old_pass && new_pass) {
        const isMatch = await verifyPassword(old_pass, current_user.password);
        if (!isMatch) {
          return res.status(400).send({ message: "Old password is incorrect" });
        }
        if (new_pass !== c_pass) {
          return res.status(400).send({ message: "Passwords do not match" });
        }
        const newpass = await hashPassword(new_pass);
        updates.password = newpass;
      }

      // Update only the provided fields (name, username, email)
      if (name) updates.name = name;
      if (username) updates.username = username;
      if (email) updates.email = email;

      // Update the user document in the database if there are updates
      if (Object.keys(updates).length > 0) {
        await User.updateOne({ _id: current_user._id }, { $set: updates });
      }

      // Fetch the updated user document
      const updatedUser = await User.findOne({ _id: current_user._id });

      // Update the session user with the updated details
      req.session.user = updatedUser;
      // Update successful response
      res.status(200).json({ message: "Update successful!" });
    } catch (error) {
      console.error(error);
      res.status(500).send({ message: "Internal Server Error" });
    }
  }
);

module.exports = { userRoutes };
