const { Router } = require("express");
const { requireLogin } = require("../middleware/authMiddleware.js");
const { Category } = require("../models/category.js");
const { Nomination } = require("../models/nomination.js");
const { Vote } = require("../models/vote.js");
const { event_profile } = require("../models/config.js");
const {loadConfig} = require("../utilities/config.js");
const { serialize } = require('bson');
const config = loadConfig();
const adminRoutes = Router();

adminRoutes.post("/category", requireLogin, async (req, res) => {
  const { name } = req.body;
  let category = name;
  const results = [];
  let exists;
  try {
    exists = await Category.find({ name: name });
  } catch (error) {
    console.error(error);
    return res.status(500).send("Internal Server Error");
  }

  if (!exists[0]) {
    let new_category = new Category({ name: name });
    try {
      await new_category.save();
      results.push({
        category,
        status: "success",
        message: "Category added Successfully!",
      });
    } catch (error) {
      console.error(error);
      return res.status(500).send("Internal Server Error");
    }
  } else {
    results.push({
      category,
      status: "error",
      message: "Category already exists",
    });
    return res.status(401).json(results);
  }

  res.status(200).json(results);
});

adminRoutes.delete("/category/:id", requireLogin, async(req,res)=>{
  const id = req.params.id;
  try{
    const update = await Category.deleteOne({_id:id});
    if(update.acknowledged == true){
      return res.status(201).json(`${id} deleted succesfully`)
    }else{
      return res.status(500).json(`nonexistent`);
    }}catch(e){
      res.send(500).json(`operation Failed`)
    }
  console.log(update);
})

// Route to update nominee approval state or delete nominee if purge
adminRoutes.post("/nominee_approval/:state", requireLogin, async (req, res) => {
  try {
    const state = req.params.state;
    const { nominee } = req.body;

    if (!nominee) {
      return res.status(400).json({ message: "Nominee ID is required" });
    }

    if (state === "purge") {
      const deleteResult = await Nomination.deleteOne({ _id: nominee });

      if (deleteResult.deletedCount === 0) {
        return res.status(404).json({ message: "Nominee not found" });
      }

      return res.status(200).json({ message: "Nominee deleted successfully" });
    } else {
      const isApproved = state === "approve";
      const updateResult = await Nomination.updateOne({ _id: nominee }, { isApproved });

      if (updateResult.nModified === 0) {
        return res.status(404).json({ message: "Nominee not found" });
      }

      return res.status(200).json({ message: `Nominee approval state updated to ${state}` });
    }
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Error updating nominee approval state" });
  }
});

adminRoutes.get("/nominees", requireLogin, async (req, res) => {
  try {
    const nominees = await Nomination.find().populate("category").lean();
    const formattedNominees = nominees.map(nominee => ({
      ...nominee,
      category: nominee.category ? nominee.category.name : "No Category",
    }));

    res.status(200).send({ nominees: formattedNominees });
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Error fetching nominees" });
  }
});

adminRoutes.get("/dashboard/:category", async (req, res) => {
  try {
    const category = req.params.category;
    const Cat = await Category.findById(category);
    const votes = await Vote.countDocuments({ category: category });

    const pipeline = [
      {
        $match: {
          category: Cat._id,
          isApproved: true,
        },
      },
      {
        $lookup: {
          from: "votes",
          localField: "_id",
          foreignField: "nominee",
          as: "votes",
        },
      },
      {
        $addFields: {
          category: { $ifNull: ["$category", "No Category"] },
        },
      },
      {
        $lookup: {
          from: "categories",
          localField: "category",
          foreignField: "_id",
          as: "category",
        },
      },
      {
        $unwind: {
          path: "$category",
          preserveNullAndEmptyArrays: true,
        },
      },
      {
        $addFields: {
          category: "$category.name",
        },
      },
      {
        $addFields: {
          votes: { $size: "$votes" },
        },
      },
      {
        $project: {
          _id: 1,
          name: 1,
          imageUrl: 1,
          category: 1,
          votes: 1,
          code: 1,
        },
      },
      {
        $sort: {
          votes: -1, // Sort by votes in descending order
        },
      },
    ];

    const Candidates = await Nomination.aggregate(pipeline);

    res.render("dashboard", { Candidates, Cat, votes });
  } catch (error) {
    console.error("Error fetching dashboard data:", error);
    res.status(500).send("Internal Server Error");
  }
});

// Route to update nominations status
adminRoutes.post("/nomination/:status", async (req, res) => {
  const state = req.params.status === "true"; // Convert to boolean
  const event = await config; 
  
  try {
    const updateResult = await event_profile.updateOne(
      {name:event.name},
      { $set: { nominations: state } }
    );

    if (updateResult.modifiedCount > 0) {
      return res.status(201).json({status:true});
    } else {
      return res.status(404).json({status:false});
    }
  } catch (err) {
    console.error("Error updating nominations:", err);
    return res.status(500).json("Unable to update Nominations");
  }
});

// Route to update voting status
adminRoutes.post("/voting/:status", async (req, res) => {
  const state = req.params.status === "true"; // Convert to boolean
  const event = await config; 
  
  try {
    const updateResult = await event_profile.updateOne(
      {name:event.name},
      { $set: { voting: state } }
    );
    //console.log(updateResult)
    if (updateResult.modifiedCount > 0) {
      return res.status(201).json({status:true});
    } else {
      return res.status(404).json({status:false})
    }
  } catch (err) {
    console.error("Error updating voting:", err);
    return res.status(500).json("Unable to update Voting");
  }
});

// Route to update voting status
adminRoutes.post("/vote_results/:status", async (req, res) => {
  const state = req.params.status === "true"; // Convert to boolean
  const event = await config; 
  
  try {
    const updateResult = await event_profile.updateOne(
      {name:event.name},
      { $set: { public_viewing: state } }
    );
    //console.log(updateResult)
    if (updateResult.modifiedCount > 0) {
      return res.status(201).json({status:true});
    } else {
      return res.status(404).json({status:false})
    }
  } catch (err) {
    console.error("Error updating voting:", err);
    return res.status(500).json("Unable to update Voting");
  }
});

module.exports = { adminRoutes };
