const { Vote } = require("../models/vote.js");
const sendMail = require("../utilities/mail.js");
const { User } = require("../models/users.js");
const { Nomination } = require("../models/nomination.js");

const createVoteModel = async (
  userDetails,
  code,
  numberOfVotes,
  batchSize = 1000
) => {
  const saveBatch = async (batch) => {
    try {
      await Vote.insertMany(batch); // Assuming Mongoose or similar ORM
    } catch (error) {
      console.error("Error saving a batch of votes:", error);
      throw new Error("Failed to save batch of votes");
    }
  };

  const batches = [];
  let voteIndex = 0;
  for (let i = 0; i < numberOfVotes; i += batchSize) {
    const batch = [];
    for (let j = i; j < Math.min(i + batchSize, numberOfVotes); j++) {
      batch.push({
        voter: userDetails,
        nominee: code.trim(),
        category: userDetails.category.trim(),
        index: ++voteIndex, // Increment the vote index for each vote
      });
    }
    batches.push(batch);
  }

  try {
    await Promise.all(batches.map(saveBatch));
    const Nominee = await Nomination.findById(code);
    const users = await User.find();

    if (users) {
      users.forEach(async (user) => {
        const message = {
          subject: "New Votes Received on Voting Portal",
          text: `Hi ${user.name},
        
        ${numberOfVotes} votes have been cast by ${userDetails.contact} for ${Nominee.name}.
        
        Best regards,
        Your Voting Portal Team`,
          html: `<p>Hi ${user.name},</p>
                 <p><strong>${numberOfVotes}</strong> votes have been cast by <strong>${userDetails.contact}</strong> for <strong>${Nominee.name}</strong>.</p>
                 <p>Best regards,<br>
                 Your Voting Portal Team</p>`,
        };

        await sendMail(user.email, message);
      });
    }
    return "All votes have been saved successfully.";
  } catch (error) {
    console.error("Error saving votes:", error);
    throw new Error("Failed to save votes");
  }
}; 

const initiateVoting = async (req, res) => {
  const { reference, transaction, trxref, contact, votes, code, category } =
    req.body;

  if(!reference||!transaction ||!trxref||!contact||!votes||!code||!category){
    return res.status(400).json({
      status: "Failed",
      message: 'All parameters needed'
    });
  }
    
  const voteData = {
    reference,
    transaction,
    trxref,
    contact,
    category,
  };

  try {
    // Check for existing votes with the same reference
    const existingVotes = await Vote.find({ "voter.reference": reference });

    if (existingVotes.length > 0) {
      return res
        .status(400)
        .json({ error: "Cannot duplicate votes from the same transaction." });
    }

    // No existing votes found, proceed to create new votes
    const response = await createVoteModel(voteData, code, votes);
    return res.status(201).json({ message: response });
  } catch (error) {
    console.error("Error initiating voting:", error);
    return res.status(500).json({ error: "Failed to initiate voting process" });
  }
};

module.exports = { initiateVoting };
