const path = require("path"); // Import the path module

const express = require("express");
const session = require("express-session");
//const mongoose = require("mongoose")
const app = express();
const { voteRoutes } = require("./routes/voteRoutes.js");
const { userRoutes } = require("./routes/userRoutes.js");
const { nominationRoutes } = require("./routes/nominationRoutes.js");
const { Category } = require("./models/category.js");
const { Nomination } = require("./models/nomination.js");
const { adminRoutes } = require("./routes/adminRoutes.js");
const { requireLogin } = require("./middleware/authMiddleware.js");
//const { log } = require("console")
const { loadConfig } = require("./utilities/config.js");
const compression = require("compression");
const MongoStore = require("connect-mongo");
const axios = require('axios');
const { categoryRoutes } = require("./routes/categoryRoutes.js");
const { event_profile } = require("./models/config.js");

const { analytics } = require("./utilities/analytics.js");
const fs = require("fs");
const logger = require("./utilities/logger.js");
const { getConfig, validKey } = require("./utilities/appConfig.js");

// Simple log parser function
const parseLogFile = (logFilePath) => {
  try {
    const logContent = fs.readFileSync(logFilePath, 'utf8');
    const lines = logContent.split('\n').filter(line => line.trim() !== '');
    
    return lines.map((line, index) => {
      // Simple parsing for each line
      const timestamp = new Date().toISOString();
      let level = 'info';
      let message = line;
      
      // Try to extract level from line
      const levelMatch = line.match(/\[(error|warn|warning|info|debug|verbose|silly)\]/i);
      if (levelMatch) {
        level = levelMatch[1].toLowerCase();
        message = line.replace(levelMatch[0], '').trim();
      }
      
      // Try to extract timestamp
      const timestampMatch = line.match(/^(\d{4}-\d{2}-\d{2}[T\s]\d{2}:\d{2}:\d{2})/);
      const extractedTimestamp = timestampMatch ? timestampMatch[1] : timestamp;
      
      return {
        id: index + 1,
        raw: line,
        timestamp: extractedTimestamp,
        level,
        message,
        meta: {},
        format: 'text'
      };
    }).reverse(); // Most recent first
  } catch (error) {
    console.error('Error reading log file:', error);
    return [];
  }
};

// Load config from database
// (moved to utilities/appConfig.js to avoid circular dependencies)

const port = process.env.port || 9999;

// Connecting to the database
// (moved to utilities/appConfig.js to avoid circular dependencies)

// Configure Express to handle JSON data
app.use(express.json());

// Configure Express to handle URL-encoded data
app.use(express.urlencoded({ extended: true }));

// Set the views directory
app.set("views", path.join(__dirname, "../client"));
app.set("view engine", "ejs");
app.use(express.static(path.join(__dirname, "../client/public")));
//app.use(express.static(path.join(__dirname,"../client/public/Images")));

//enable compression
app.use(compression());

//cross origin management
//app.use(cors());
const mongoURi =
  process.env.MONGODB_CONNECTION_STRING ||
  process.env.MONGODB_CONNECTION_STRING_local;

//session management
app.use(
  session({
    secret: "ustpollzzz",
    resave: false,
    saveUninitialized: true,
    store: MongoStore.create({
      mongoUrl: mongoURi,
      dbName: process.env.DATABASE_NAME,
      collectionName: "sessions",
    }),
    cookie: {
      maxAge: 1000 * 60 * 60 * 1, // 1 day
    },
  })
);

app.use("/vote", voteRoutes);
app.use("/u", userRoutes);
app.use("/nominations", nominationRoutes);
app.use("/admin", requireLogin, adminRoutes);
app.use("/category", categoryRoutes);

app.get("/user", (req, res) => {
  res.render("index");
});

app.get("/", async (req, res) => {
  try {
    logger.info("Home page accessed");
    const profile = await event_profile.findOne();
    if (!profile) {
      logger.warn("Profile not found, redirecting to config");
      return res.redirect("/config");
    }
    const event = profile; // Use the profile directly as the event config
    const analyticsData = await analytics();
    const categories = await Category.find();
    logger.info(`Categories page rendered with ${categories.length} categories`);

    res.render("categories", { categories, event, analyticsData });
  } catch (error) {
    logger.error("Error fetching categories: " + error.message);
    res.status(500).send("Internal Server Error");
  }
});

app.get("/search", async (req, res) => {
  const { query } = req.query;
  logger.info(`Search request received for query: "${query}"`);

  if (!query) {
    logger.warn("Search request without query parameter");
    return res.status(400).json({ message: "Query parameter is required" });
  }

  try {
    const results = await Nomination.find({
      $or: [
        { code: { $regex: query, $options: "i" } },
        { name: { $regex: query, $options: "i" } },
      ],
    }).exec();

    if (results.length === 0) {
      logger.info(`No search results found for query: "${query}"`);
      return res.status(404).json({ message: "No nominees found" });
    }
    
    logger.info(`Found ${results.length} search results for query: "${query}"`);
    const sortedResults = results.sort((a, b) => {
      const aMatch = Math.max(
        a.code.toLowerCase().includes(query.toLowerCase()) ? 1 : 0,
        a.name.toLowerCase().includes(query.toLowerCase()) ? 1 : 0
      );
      const bMatch = Math.max(
        b.code.toLowerCase().includes(query.toLowerCase()) ? 1 : 0,
        b.name.toLowerCase().includes(query.toLowerCase()) ? 1 : 0
      );
      return bMatch - aMatch;
    });

    res.status(200).json(results);
  } catch (error) {
    logger.error("Error searching nominees: " + error.message);
    res.status(500).json({ message: "Error searching nominees" });
  }
});

app.get("/config",requireLogin, async (req, res) => {
  try {
    logger.info("Config setup page accessed");
    res.render("setup");
  } catch (error) {
    logger.error("Error rendering setup page: " + error.message);
    res.status(500).send("Internal Server Error");
  }
});

app.post("/config/:key", async (req, res) => {
  const key = req.params.key;
  const { name, picture, logo, code_prefix, rate, keyy, income, closingDateTTime } = req.body;
  
  logger.info(`Configuration setup attempt with key: ${key}`);

  // Basic validation
  const missingFields = [];
  if (!name) missingFields.push("name");
  if (!picture) missingFields.push("picture");
  if (!logo) missingFields.push("logo");
  if (!code_prefix) missingFields.push("code_prefix");
  if (!rate) missingFields.push("rate");
  if (!keyy) missingFields.push("keyy");
  if (!income) missingFields.push("income");
  if (!closingDateTTime) missingFields.push("closingDateTTime");
  
  if (missingFields.length > 0) {
    logger.warn(`Configuration setup failed - missing fields: ${missingFields.join(", ")}`);
    return res.status(400).json({ message: `Missing fields: ${missingFields.join(", ")}` });
  }

  const model = {
    name: name,
    picture: picture,
    logo: logo,
    code_prefix: code_prefix,
    rate: rate,
    key: keyy,
    income: income,
    closingDateTTime: new Date(closingDateTTime),
  };

  if (key === "ust_polls") {
    try {
      const exists = await event_profile.findOne();
      if (exists) {
        logger.warn("Configuration setup failed - profile already exists");
        return res
          .status(400)
          .json({ message: "Profile setup already exists" });
      }

      // Create profile in local database
      const profile = new event_profile(model);
      await profile.save();
      
      logger.info(`Configuration created successfully for event: ${name}`);

      // Prepare data for admin system
      const adminProfileData = {
        name: name,
        code_prefix: code_prefix,
        rate: parseFloat(rate),
        connectionString: mongoURi, // Use your MongoDB connection string
        voting: true, // Set to true since voting will be active
        dbName: process.env.DATABASE_NAME || 'ustpolls',
        link: req.protocol + '://' + req.get('host'), // Current domain
        closingDateTime: new Date(closingDateTTime),
        isActive: true,
        description: `Event: ${name}`,
        status: 'active'
      };

      // Make request to admin system
      try {
        logger.info("Attempting to create profile in admin system");
        
        const adminResponse = await axios.post(
          'https://admin.ustpolls.com/api/events/create'|| 'https://ustpolls-admin.onrender.com/api/events/create',
          adminProfileData,
          {
            headers: {
              'Content-Type': 'application/json',

            },
            timeout: 30000 // 30 second timeout
          }
        );

        if (adminResponse.status === 200 || adminResponse.status === 201) {
          logger.info(`Profile successfully created in admin system: ${adminResponse.data.message || 'Success'}`);
        } else {
          logger.warn(`Admin system responded with status ${adminResponse.status}: ${adminResponse.data.message || 'Unknown'}`);
        }
        
      } catch (adminError) {
        // Log the error but don't fail the main creation
        logger.error(`Failed to create profile in admin system: ${adminError.message}`);
        
        if (adminError.response) {
          logger.error(`Admin API Error - Status: ${adminError.response.status}, Data: ${JSON.stringify(adminError.response.data)}`);
        } else if (adminError.request) {
          logger.error('Admin API Error - No response received from admin system');
        } else {
          logger.error(`Admin API Error - Request setup: ${adminError.message}`);
        }
        
        // Continue with success response since local creation succeeded
        logger.info("Continuing with local profile creation success despite admin system error");
      }

      return res.status(201).json({ 
        success: true, 
        message: "Config Created Successfully",
        adminSync: true // Indicates attempt was made to sync with admin
      });
      
    } catch (error) {
      logger.error("Error creating configuration: " + error.message);
      return res
        .status(500)
        .json({ success: false, message: "Unable to Create Configuration" });
    }
  } else {
    logger.warn(`Configuration setup failed - incorrect key: ${key}`);
    return res.status(400).json({ success: false, message: "Incorrect Key" });
  }
});

app.get("/logs/:key", async (req, res) => {
  try {
    logger.info(`Log access attempt with key: ${req.params.key}`);
    const config = await getConfig();
    const key =  config.key; // Fallback to 'rss' if no config
    
    if (req.params.key !== key) {
      logger.warn(`Unauthorized log access attempt with key: ${req.params.key}`);
      return res.status(401).json({ message: "Unauthorized Access" });
    }

    logger.info("Log access granted - serving logs");
    const logFilePath = path.join(__dirname, "../combined.log");
    
    // Check if log file exists
    if (!fs.existsSync(logFilePath)) {
      logger.warn("Log file not found");
      return res.render("logs", { logs: [] });
    }

    // Parse the log file
    const logs = parseLogFile(logFilePath);
    
    // Limit to last 1000 entries for performance
    const limitedLogs = logs.slice(0, 1000);
    
    logger.info(`Serving ${limitedLogs.length} log entries`);
    return res.render("logs", { logs: limitedLogs });
    
  } catch (error) {
    logger.error("Error reading logs: " + error.message);
    return res.status(500).json({ message: "Error reading logs" });
  }
});

// Route for downloading raw log file
app.get("/logs/:key/download", async (req, res) => {
  try {
    const config = await getConfig();
    const key =  config.key ; // Fallback to 'rss' if no config
    
    if (req.params.key !== key) {
      return res.status(401).json({ message: "Unauthorized Access" });
    }

    const logFilePath = path.join(__dirname, "../combined.log");
    
    if (!fs.existsSync(logFilePath)) {
      return res.status(404).json({ message: "Log file not found" });
    }

    return res.download(logFilePath, "ustpolls-logs.log");
    
  } catch (error) {
    console.error("Error downloading logs:", error);
    return res.status(500).json({ message: "Error downloading logs" });
  }
});

// Start the server
app.listen(port, () => {
  logger.info(`🚀 USTPolls server started on port ${port}`);
  logger.info("=".repeat(50));
  logger.info("Server initialization complete");
  console.log(`Server is listening on port ${port}`);
});
